function price = priceCallByHeatEquationExplicit( K, T, S0, r, sigma, N, M, nSds )
%EULERFORCALL Use the explicit method to price a call option
%   by solving the heat equation in negative time
%   N is the number of time steps
%   M is the number of steps in the x direction
%   nSds is the number of standard deviations either side of S0 to use
%        when defining the boundary of the grid.
%
%   We make the coordinate changes x=-(r-0.5 sigma^2)t + log S
%                                  W=e^{-rt} V
%   to transform the BS PDE into the heat equation in negative time
assert( rem(M,2)==0); % M must be even

x0 = log( S0 );
xMin = x0 - nSds*sqrt(T)*sigma;
xMax = x0 + nSds*sqrt(T)*sigma;

dt = T/N;
dx = (xMax-xMin)/M;

iMin = 1;
iMax = N+1;
jMin = 1;
jMax = M+1;
x = xMin:dx:xMax;
t = 0:dt:T;

lambda = 0.5*sigma^2 * dt/(dx)^2;

% Use boundary condition to create vector currW
currW=max(exp(-0.5*sigma^2*T + x) - exp(-(r*T))*K,0); 

for i=iMax:-1:iMin+1    
    % Use recurrence for all points except jMin and jMax
    currW(jMin+1:jMax-1)=      lambda*currW((jMin):(jMax-2)) ...
                        +(1-2*lambda)*currW((jMin+1):(jMax-1)) ...
                        +      lambda*currW((jMin+2):(jMax));
    % Use boundary conditions for jMin and jMax
    currW(jMin)=0; 
    currW(jMax)=exp(-0.5*sigma^2 * t(i) + x(jMax))- exp(-r*T)*K;                    
end    

price = currW(iMin,jMin+M/2);

end

