% Suppose we sell a call option to a customer for the cost 
%  chargeToCustomer
% and the option has strike K1 and maturity T.
% The market is assumed to follow a model with parameters S0, r, mu, sigma
%
% We gamma hedge using a second call with strike K2 at nSteps intervals up to time sellTime
% and then disolve at current market prices (assumed to be Black Scholes
% prices)
%
% We run the simulation nPaths times.
function [ marketValue] = simulateGammaHedging(  chargeToCustomer, K1, K2, T, S0, r, mu, sigma, nPaths, nSteps, sellTime  )

dt = sellTime/nSteps;
times = dt:dt:sellTime;
paths = generateBSPaths( sellTime, S0, mu, sigma, nPaths, nSteps );

[~,delta1, gamma1] = blackScholesCallPrice(K1,T,S0,r,sigma);    
[p2,delta2, gamma2] = blackScholesCallPrice(K2,T,S0,r,sigma);    
option2Quantity = max(min(gamma1./gamma2,100),-100);
stockQuantity = delta1 - option2Quantity .* delta2;
stockCost = stockQuantity .* S0;
optionCost = option2Quantity .* p2;
bankBalance = chargeToCustomer -stockCost-optionCost;

for t=1:nSteps
    
    S = paths(:,t);
    timeToMaturity = T-times(t);
    [p1,delta1,gamma1] = blackScholesCallPrice(K1,timeToMaturity,S,r,sigma);
    [p2,delta2,gamma2] = blackScholesCallPrice(K2,timeToMaturity,S,r,sigma);    
    newOption2Quantity = max(min(gamma1./gamma2,100),-100);
    newStockQuantity = delta1 - newOption2Quantity .* delta2;
    
    stockCost = (newStockQuantity - stockQuantity).* S;
    optionCost = (newOption2Quantity - option2Quantity).* p2;
    bankBalance = exp(r*dt)*bankBalance - stockCost - optionCost;
    
    stockQuantity = newStockQuantity;
    option2Quantity = newOption2Quantity;
    
    marketValue = bankBalance + stockQuantity.*S - p1 + option2Quantity.*p2;
                  
end    

end

