% Suppose we sell a call option to a customer for the cost 
%  chargeToCustomer
% and the option has strike K and maturity T.
% The market is assumed to follow a model with parameters S0, r, mu, sigma
% We delta hedge the at nSteps intervals up to maturity and compute the
% final bank balance.
%
% We run the simulation nPaths times.
function [ finalBankBalance] = simulateDeltaHedgingWithTxCosts(  chargeToCustomer, K, T, S0, r, mu, sigma, nPaths, nSteps, epsilon  )

dt = T/nSteps;
times = dt:dt:T;
paths = generateBSPaths( T, S0, mu, sigma, nPaths, nSteps );

[~,delta] = blackScholesCallPrice(K,T,S0,r,sigma);    
stockQuantity = delta;
cost = computeCost( stockQuantity, S0, epsilon);
bankBalance = chargeToCustomer-cost;

for t=1:(nSteps-1)    
    S = paths(:,t);
    timeToMaturity = T-times(t);
    [~,delta] = blackScholesCallPrice(...
        K,timeToMaturity,S,r,sigma);
    newStockQuantity = delta;
    amountToBuy = newStockQuantity - stockQuantity;
    cost = computeCost( amountToBuy, S, epsilon);
    bankBalance = exp(r*dt)*bankBalance - cost;
    
    stockQuantity = newStockQuantity;
end    

S = paths(:,nSteps);
stockValue = -computeCost( -stockQuantity, S, epsilon);
liability = max(S-K, 0);

bankBalance = exp(r*dt)*bankBalance + stockValue - liability;

finalBankBalance = bankBalance;

end

% Compute the cost of buying quantity of the stock (or the
% return of selling them if quantity is negative)
function v=computeCost( quantity, stockPrice, epsilon ) 
price = stockPrice - (quantity<0) .* epsilon .* stockPrice;
v = quantity.*price;
end

