import math
import numpy
import copy
import numpy.linalg
import matplotlib.pyplot as plt




class Line2D:
    """Represents a single line segment joining two points"""

    def __init__(self, p1=numpy.array([0,0]), p2=numpy.array([1,1])):
        self.p1 = p1
        self.p2 = p2

    def draw(self):
        xVals = [self.p1[0],self.p2[0]]
        yVals = [self.p1[1],self.p2[1]]
        lines = plt.plot(xVals, yVals, 'k-')
        plt.setp(lines, linewidth=2)

    def length(self):
        p1 = self.p1;
        p2 = self.p2;
        return math.sqrt((p1[0]-p2[0])**2 + (p1[1]-p2[1])**2)

    def __str__(self):
        return 'A line from {} to {}'.format(self.p1, self.p2)



class Graphics2D:
    """An object that groups together shapes that can then be displayed using matplotlib"""

    def __init__(self):
        self.elements = []

    def add(self, element):
        self.elements.append(element)

    def show(self, large=False, file=None):
        plt.axis('off')
        plt.axes().set_aspect('equal', 'datalim')
        for element in self.elements:
            element.draw()
        if large:
            # get current figure and set its size
            fig = plt.gcf()
            fig.set_size_inches(10, 10)
        if not file is None:
            plt.savefig(file)
        else:
            plt.show()

    def length(self):
        ret = 0.0
        for element in self.elements:
            ret+=element.length()
        return ret

    def __str__(self):
        return "A Graphics2D. Contents are: "+(", ".join(map(str,self.elements)))


class Circle2D:
    """Represents a circle"""

    def __init__(self):
        self.center = numpy.array([0,0])
        self.r = 1

    def circumference(self):
        return self.r * 2* math.pi

    def area(self):
        return (self.r ** 2 ) *math.pi

    def __str__(self):
        return 'A circle centre {} and radius {}'.format(self.center, self.r)

    def length(self):
        return self.circumference()

    def _pointAtAngle(self, theta):
        return [self.center[0] + math.cos(theta)*self.r,
                self.center[1] + math.sin(theta)*self.r]

    def draw(self):
        n = 100;
        theta = 0;
        last_point = self._pointAtAngle(theta)
        step = 2*math.pi/n;
        while theta< 2*math.pi:
            theta += step
            point = self._pointAtAngle(theta)
            line = Line2D()
            line.p1 = last_point
            line.p2 = point
            line.draw()
            last_point = point


class Isometry:
    """Isometry of R^3"""

    def __init__(self):
        self._u = numpy.array([[1,0,0],[0,1,0],[0,0,1]])
        self._v = numpy.array([0,0,0])

    def transform_point(self, point):
        # Apply the transformation to the point
        return numpy.dot(self._u,point) + self._v;

    def transform_vector(self, vector):
        # Apply the transformation to a vector
        return numpy.dot(self._u, vector);


    @property
    def u(self):
        """Unitary matrix"""
        return self._u

    @u.setter
    def u(self, value):
        value = numpy.array(value)
        product = numpy.dot(value,value.transpose())
        assert numpy.linalg.norm(product - numpy.identity(3))<0.0001, \
            "u must be a unitary matrix"
        self._u = value

    @property
    def v(self):
        """Offset vector"""
        return self._v

    @v.setter
    def v(self, value):
        self._v = numpy.array(value)

    @staticmethod
    def rotation(x_angle=0.0, y_angle=0.0, z_angle=0.0):
        """Creates a transformation representing a rotation through
           the given angle about each axis in turn"""
        i = Isometry()
        r_z = numpy.array([[math.cos(z_angle), math.sin(z_angle), 0],
                           [-math.sin(z_angle), math.cos(z_angle), 0],
                            [0, 0, 1]])
        r_x = numpy.array([[1, 0, 0],
                          [0, math.cos(x_angle), math.sin(x_angle)],
                          [0, -math.sin(x_angle), math.cos(x_angle)]])
        r_y = numpy.array([[math.cos(y_angle), 0, -math.sin(y_angle)],
                          [0, 1, 0],
                          [math.sin(y_angle), 0, math.cos(y_angle)]])
        i.u = numpy.dot(numpy.dot(r_z, r_y), r_x)
        return i

    @staticmethod
    def translation(v):
        """Creates a matrix representing a translation"""
        i = Isometry()
        i.v = v
        return i


class Viewpoint:
    """Represents the position of an artists eye and the coordinate system they are using for the canvas"""

    def __init__(self):
        self.eye = numpy.array([0,-1,0])
        self.canvasOrigin = numpy.array([0,0,0])
        self.canvasX = numpy.array([1,0,0])
        self.canvasY = numpy.array([0,0,1])

    def _perp(self):
        unnormalized = numpy.cross(self.canvasX, self.canvasY)
        return unnormalized/numpy.linalg.norm(unnormalized)

    def _project(self, coords_3d):
        """Project the point orthogonally onto the canvas, returns canvas coordinates of point and signed distance"""
        offset = coords_3d-self.canvasOrigin
        x_vec = self.canvasX
        y_vec = self.canvasY
        p = self._perp()
        distance = numpy.dot(offset, p)
        canvas_offset = offset - distance*p
        canvas_x = numpy.dot( canvas_offset, x_vec )/numpy.dot( x_vec, x_vec)
        canvas_y = numpy.dot( canvas_offset, y_vec )/numpy.dot( y_vec, y_vec)
        return numpy.array([canvas_x,canvas_y]), distance

    def coords_2d( self, coords_3d ):
        eye_proj, eye_distance = self._project( self.eye )
        point_proj, point_distance = self._project( coords_3d )
        total_distance = eye_distance - point_distance
        canvas_point = -eye_proj * point_distance/total_distance + point_proj*eye_distance/total_distance
        return canvas_point

    def transform(self, transformation: Isometry):
        other = copy.deepcopy(self)
        other.eye = transformation.transform_point(self.eye)
        other.canvasOrigin = transformation.transform_point(self.canvasOrigin)
        other.canvasX = transformation.transform_vector( self.canvasX )
        other.canvasY = transformation.transform_vector(  self.canvasY )
        return other


class Line3D:

    def __init__(self, p1=numpy.array([0,0,0]), p2=numpy.array([1,0,0])):
        self.p1 = p1
        self.p2 = p2

    def draw(self, viewpoint : Viewpoint, graphics: Graphics2D):
        c1 = viewpoint.coords_2d(self.p1)
        c2 = viewpoint.coords_2d(self.p2)
        l2 = Line2D( c1, c2)
        graphics.add(l2)

    def transform(self, transformation: Isometry):
        other = copy.deepcopy(self)
        other.p1 = transformation.transform_point(self.p1)
        other.p2 = transformation.transform_point(self.p2)
        return other

    def __repr__(self):
        return repr_data_object(self)


def _compute_unit_cube_edges():
    """Compute a list of pairs representing the edges in the unit cube """
    ret = []
    offsets = [numpy.array([i, j, k]) for i in range(0, 2) for j in range(0, 2) for k in range(0, 2)];
    for i1 in range(0, len(offsets)):
        for i2 in range(i1 + 1, len(offsets)):
            p1 = offsets[i1]
            p2 = offsets[i2]
            diff = p1 - p2
            if abs(numpy.dot(diff, diff) - 1.0) < 0.001:
                ret.append((p1, p2))
    return ret


class Cube:
    """A cube with the given side length and edges in the directions v1, v2 and v1xv2"""

    _unit_cube_edges = _compute_unit_cube_edges()
    """Pairs of triples indicating the edges of a cube"""

    def __init__(self):
        self._origin = numpy.array([0,0,0])
        self._v1 = numpy.array([1,0,0])
        self._v2 = numpy.array([0,1,0])
        self._v3 = numpy.array([0,0,1])
        self.side_length = 1

    def _vertex(self, offsets):
        return self._origin + self.side_length*(offsets[0]*self._v1 + offsets[1]*self._v2 + offsets[2]*self._v3)

    def _edges(self):
        edges = []
        for p1,p2 in Cube._unit_cube_edges:
            v1 = self._vertex(p1)
            v2 = self._vertex(p2)
            line3d = Line3D(v1,v2)
            edges.append(line3d)
        return edges

    def draw(self, viewpoint: Viewpoint, graphics: Graphics2D):
        for edge in self._edges():
            edge.draw(viewpoint, graphics)

    def transform(self, transformation : Isometry):
        other = copy.deepcopy(self)
        other._v1 = transformation.transform_vector( self._v1)
        other._v2 = transformation.transform_vector( self._v2)
        other._v3 = transformation.transform_vector( self._v3)
        other._origin = transformation.transform_point(self._origin);
        return other

    def __repr__(self):
        return repr_data_object(self)


def repr_data_object( obj ):
    """Print out a detailed representation of a data object"""
    ret = "[ "+str(type(obj)) +": "
    for name, val in vars( obj ).items():
        if not callable(val):
            ret += name + "="
            ret += repr(val)
            ret += " "
    ret += "]"
    return ret
